#include <Arduino.h>
#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <FS.h>
#include <LITTLEFS.h>
#include <PubSubClient.h>

// WiFi Credentials
const char* ssid = "YourSSID";
const char* password = "YourPassword";

// MQTT Broker Settings
const char* mqttServer = "192.168.20.31";
const int mqttPort = 1883;
const char* mqttUser = "user";
const char* mqttTopic = "/REMOTESW/ANTENNA";

// GPIO Pins for Antennas
const int antennaPins[] = {13, 12, 14, 27, 26, 25, 33, 32};
const char* antennaNames[] = {"01", "02", "03", "04", "05", "06", "07", "08"};

// Temperature Sensor (DS18B20) on GPIO2
#define ONE_WIRE_BUS 2
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature sensors(&oneWire);
float temperatureC = 0.0;

// Web Server and MQTT Client
AsyncWebServer server(80);
WiFiClient espClient;
PubSubClient client(espClient);

// Function to Read Temperature from DS18B20 Sensor
void readTemperature() {
    sensors.requestTemperatures(); // Request temperature from sensor
    temperatureC = sensors.getTempCByIndex(0); // Get temperature in Celsius
}

// Function to Set GPIO Pins Based on Selected Antenna and Publish to MQTT
void setAntenna(int index) {
    for (int i = 0; i < 8; i++) {
        digitalWrite(antennaPins[i], (i == index) ? HIGH : LOW); // Set selected antenna HIGH, others LOW
    }
    if (client.connected()) {
        client.publish(mqttTopic, antennaNames[index]); // Publish selected antenna to MQTT topic
    }
}

// Handle Web Requests to Select an Antenna
void handleRequest(AsyncWebServerRequest *request) {
    if (request->hasParam("antenna")) {
        int index = request->getParam("antenna")->value().toInt(); // Get antenna index from request
        if (index >= 0 && index < 8) {
            setAntenna(index); // Set the selected antenna
        }
    }
    request->send(LITTLEFS, "/index.html", "text/html"); // Serve the HTML page from LITTLEFS
}

// Setup WiFi Connection
void setupWiFi() {
    WiFi.begin(ssid, password);
    while (WiFi.status() != WL_CONNECTED) {
        delay(500); // Wait for WiFi connection
        Serial.println("No WiFi!");
    }
}

// Setup MQTT Connection
void setupMQTT() {
    client.setServer(mqttServer, mqttPort);
    client.connect("ESP32Client", mqttUser, ""); // Connect to MQTT broker
}

// Setup Function
void setup() {
    Serial.begin(115200); // Start serial communication
    Serial.printf("Setup WiFi\r\n");
    setupWiFi(); // Connect to WiFi
    setupMQTT(); // Connect to MQTT broker
    
    if (!LITTLEFS.begin()) {
        Serial.println("Failed to mount file system"); // Error message if file system fails
        return;
    }
     Serial.printf("Mounted LittleFS\r\n");
   
    sensors.begin(); // Initialize temperature sensor
    Serial.printf("Initalized sensor\r\n");
    for (int i = 0; i < 8; i++) {
        pinMode(antennaPins[i], OUTPUT);
        digitalWrite(antennaPins[i], LOW); // Set all antennas to LOW initially
    }
    Serial.printf("Initalized gpio\r\n");
    
    server.on("/", HTTP_GET, handleRequest); // Define route for serving the webpage
    server.begin(); // Start the web server
    Serial.printf("Webserver started gpio\r\n");

}

// Main Loop Function
void loop() {
    if (!client.connected()) {
        setupMQTT(); // Reconnect to MQTT if disconnected
    }
    client.loop(); // Maintain MQTT connection
    
    static unsigned long lastTempRead = 0;
    if (millis() - lastTempRead > 30000) { // Read temperature every 30 seconds
        lastTempRead = millis();
        readTemperature();
    }
}
